Jetpack Start
=============

Jetpack Start is our attempt at exploring a better New User Experience flow when a user first sets up WordPress.

![alt tag](https://raw.github.com/Automattic/jetpack-start/master/img/jps_started.png)

It is designed to be extensible, so that hosting companies that choose to bundle it can easily modify, add, or remove any of the steps that we've included by default.

Most of the code lives in the client, and uses [ReactJS](https://github.com/facebook/react) and Facebook's [Flux Dispatcher](https://github.com/facebook/flux) to manage the flow of data and UI updates. 

This initiative is currently being led by Daniel Walmsley [@gravityrail](http://github.com/gravityrail) and Luca Sartoni [@lucasartoni](https://github.com/lucasartoni), with design by Dan Hauk [@danhauk](https://github.com/danhauk).

Previous contributors who laid the foundation are Dave Martin [@growthdesigner](http://github.com/growthdesigner) and Miguel Lezama [@lezama](http://github.com/lezama).

Pull Requests and Issues are always welcome. :)

## Integrating

This plugin publishes three hooks:
- jps_started
- jps_step_skipped
- jps_step_complete

jps_started is invoked when the user clicks the "Get Started ->" link on the front page of the wizard. The latter two "step" hooks are invoked with a string (a "slug") which names the step. 

Each "jps_step_completed" step is accompanied by a data hash, which at a minimum includes an entry called "completion", which is the % completion of the wizard. For example, a step completion hash for the "design" step might look like this:

```
$data = array(
	'themeId' => 'edit',
	'completion' => 60
)
```

An integration might look like this:

```php
<?php
/**
 * Plugin Name: My Jetpack Start Tracking Plugin
 * Plugin URI: https://github.com/someone/jetpack-start-tracker
 * Description: Tracking for Jetpack Start
 * Version: 0.1
 */

class JetpackStartTracking {
	static function track_jps_usage() {
		add_action('jps_started', array(__CLASS__, 'track_started'));
		add_action('jps_step_skipped', array(__CLASS__, 'track_step_skipped'));
		add_action('jps_step_completed', array(__CLASS__, 'track_step_completed'));
	}

	static function track_started() {
		self::record_user_event('none', 'started');
	}

	static function track_step_skipped($step_slug) {
		self::record_user_event($step_slug, 'step_skipped', array());
	}

	static function track_step_completed($step_slug, $data) {
		// note: $data is an associative array of metadata related to the step completed
		// e.g. when the "design" step is completed, data looks like: {themeId: 'the-theme-id'}
		self::record_user_event($step_slug, 'step_completed', $data);
	}

	static function record_user_event($step_slug, $event_type, $data) {
		$current_user = wp_get_current_user();
		$event = array(
			'_event_type' => 'jps_'.$event_type,
			'step' => $step_slug,
			'user_id' => $current_user->ID,
			'user_email' => $current_user->user_email,
			'_ip' => $_SERVER['REMOTE_ADDR'],
			'data' => $data
		);
		error_log("Recorded event: ".print_r($event, true));
	}
}

add_action( 'init',  array('JetpackStartTracking', 'track_jps_usage') );
}
```

## Building

```bash
cd /path/to/jetpack-start
npm install -g grunt grunt-cli # if you don't have grunt
npm install                    # install local dependencies
grunt                          # by default, this watches your JS and SCSS files for changes and automatically rebuilds the client app
```

Directory structure:

- client - this is where you come to edit javascript
  - actions - Flux actions. These are called by components, and manage server-side data updates and telling the Dispatcher that an event/update has occurred.
  - components - React components, written as JSX
  - constants - Shared JS constants
  - dispatcher - the Flux App Dispatcher
  - stores - Flux stores, which receive callbacks from the Dispatcher, modify client data state and tell subscribed components to update themselves
  - utils - currently just a wrapper for jQuery.ajax that handles the WP way of rendering JSON errors.
  - jetpack-start.js - entry point for the JS client app
  - welcome-panel.js - entry point that configures and initialises the Welcome Panel on the dashboard.
- css - edit the SCSS files in here. CSS files are generated by grunt-sass (see above)
- font - just genericons
- js - the generated client JS bundles. DO NOT MODIFY. "grunt" handles this for you (see above)

## Debugging

If you load the dashboard with the parameter "jps_reset=1", then all Jetpack Start *AND* Jetpack data will be reset. 

If you enable WP_DEBUG in wp-config.php, then you'll see some additional buttons on the wizard UI for resetting wizard progress data (just the wizard progress in this case, not Jetpack itself) and showing and hiding the spinner overlay.
